import numpy as np

def find_erosion_rate( BinFile, StationFile):
    
    '''
    Routine to compute erosion rate in a series of catchments from age data collected in a river
    stream crossing the catchments
    
    In input:
    BinFile, name of file containing the bins to be used to bin the age data
    StationFile, name of file containing, for each station (one per line),
        the name of the file containing the ages, the order of the site/catchment long the stream,
        the area of the catchment and the fertility factor
        Note that the order is negative for tributaries
        
    in output:
    erosion_out, distribution of erosion rate for each area; dimension=[nstation,nsample]
    cc_out, distribution of surface age concentrations in each area; dimension=[nstation,nbin,nsample]
    name_out, name of stations; dimension=[nstation]
    x_out, observed age distributions at each site; dimension=[nstation,nbin]
    Fertility_out, product of drainage area by surface mineral concentration factor at each site; dimension=[nstation]
    
    nstation = number of stations/sites
    nbin = number of age bins used to define the distributions
    nsample = number of samples used in the bootstrapping
    '''

    # reads bin range
    abins = np.loadtxt( BinFile)
    nbin = len( abins) - 1

    # reads Site Data
    ColumnNames=['Name', 'order', 'area', 'alpha']
    Station0 = np.genfromtxt( StationFile, dtype = None, names = ColumnNames)
    nstation0 = len( Station0)

    # selects the sites along the trunk and orders them
    Station = Station0[ Station0['order'] > 0]
    Station = np.sort( Station, order = 'order')
    nstation = len( Station)
    area = Station['area']
    alpha = Station['alpha']
    name = np.array([s.decode('utf-8') for s in Station['Name']])

    # selects the sites in tributaries and finds their "master" catchment
    StationT = Station0[ Station0['order'] < 0]
    StationT = np.flip(np.sort( StationT, order = 'order'), 0)
    nstationT = len( StationT)
    areaT = StationT['area']
    alphaT = StationT['alpha']
    nameT = np.array([s.decode('utf-8') for s in StationT['Name']])
    masterT = -StationT['order'] + 1

    for j in range(nstation):
        for i in range(nstationT):
            if np.all(Station['order']!=masterT[i]): masterT[i] = masterT[i]+1

    master = np.zeros(nstationT, dtype=int)
    for i in range(nstationT):
        for j in range(nstation):
            if masterT[i] == Station['order'][j]: master[i]=j
    area[master] += areaT
    
    # dimensions arrays (nsample is number of subsamples generated for bootstrapping;
    #       and bootstrap is the proportion of original sample in each subsample)
    
    x = np.zeros((nstation,nbin))
    xT = np.zeros((nstationT,nbin))
    nsample = 1000
    erosion = np.ones((nstation,nsample))
    erosionT = np.ones((nstationT,nsample))
    cc = np.zeros((nstation,nbin,nsample))
    ccT = np.zeros((nstationT,nbin,nsample))
    deltam = np.zeros(nstation)

    # main loop for bootstrapping
    for isample in range(nsample):

      # reads in data, performs subsampling and bins data to create age distributions
        for i in range(nstation):
            age = np.loadtxt(Station['Name'][i].decode('UTF-8'),skiprows=1)
            nage = len(age[:,0])
            agesample = np.random.choice(age[:,0],nage,replace=True)
            binsample = np.digitize(agesample, abins)
            for j in range(nbin):
                x[i,j]=(binsample==j+1).sum()+1
            xsum = x[i,:].sum()
            x[i,:] = x[i,:]/xsum

      # performs analysis as described in BG2017 to obtain erosion rate estimates and
      # age distributions in catchment
        for k in range(nbin):
            cc[0,k,isample]=x[0,k]

        for i in range(1,nstation):
            deltam[i]=1.e-6
            for k in range(nbin):
                if x[i,k]!=0: deltam[i]=max(deltam[i],(x[i-1,k]-x[i,k])/x[i,k])
                if x[i,k]!=1: deltam[i]=max(deltam[i],(x[i,k]-x[i-1,k])/(1-x[i,k]))
            for k in range(nbin):
                cc[i,k,isample]=x[i,k]+(x[i,k]-x[i-1,k])/deltam[i]

        for i in range(1,nstation):
            erosion[i,isample]=area[1]*alpha[1]/area[i]/alpha[i]*deltam[i]*np.prod((1.+deltam[1:i-1]))
            
      # reads in data, performs subsampling and bins data to create age distributions for tributaries
        for i in range(nstationT):
            ageT = np.loadtxt(StationT['Name'][i].decode('UTF-8'),skiprows=1)
            nageT = len(ageT[:,0])
            agesampleT = np.random.choice(ageT[:,0],nageT,replace=True)
            binsT = np.digitize(agesampleT, abins)
            for j in range(nbin):
                xT[i,j] = (binsT==j+1).sum()
            xTsum = xT[i,:].sum()
            xT[i,:] = xT[i,:]/xTsum

      # performs analysis as described in BG2017 to obtain erosion rate estimates and
      # age distributions in tributaries
        for i in range(nstationT):
            m = master[i]
            for k in range(nbin):
                ccT[i,k,isample] = xT[i,k]
                erosionT[i,isample] = erosion[m,isample]
                if xT[i,k]!=0:
                    erosionT[i,isample] = min(erosionT[i,isample],
                        erosion[m,isample]*alpha[m]*area[m]*cc[m,k,isample]/
                        (alphaT[i]*areaT[i]*xT[i,k]))
                if xT[i,k]!=1:
                    erosionT[i,isample] = min(erosionT[i,isample],
                        erosion[m,isample]*alpha[m]*area[m]*(1-cc[m,k,isample])/
                        (alphaT[i]*areaT[i]*(1-xT[i,k])))

    # reads in the full data to output concentrations in bins without the subsampling for bootstrapping
    for i in range(nstation):
        age = np.loadtxt(Station['Name'][i].decode('UTF-8'),skiprows=1)
        nage = len(age[:,0])
        agesample = np.random.choice(age[:,0],nage,replace=False)
        binsample = np.digitize(agesample, abins)
        for j in range(nbin):
            x[i,j]=(binsample==j+1).sum()+1
        xsum = x[i,:].sum()
        x[i,:] = x[i,:]/xsum
    for i in range(nstationT):
        ageT = np.loadtxt(StationT['Name'][i].decode('UTF-8'),skiprows=1)
        nageT = len(ageT[:,0])
        agesampleT = np.random.choice(ageT[:,0],nageT,replace=False)
        binsT = np.digitize(agesampleT, abins)
        for j in range(nbin):
            xT[i,j] = (binsT==j+1).sum()
        xTsum = xT[i,:].sum()
        xT[i,:] = xT[i,:]/xTsum

    # assembles the output (sites and subcatchments)
    erosion = np.concatenate((erosion,erosionT),axis=0)
    cc = np.concatenate((cc,ccT),axis=0)
    name = np.concatenate((name,nameT),axis=0)
    x = np.concatenate((x,xT),axis=0)
    Fertility = np.concatenate(( area*alpha, areaT*alphaT), axis=0)

    # reorders the sites to include the subcatchments
    order = np.concatenate((Station['order'],-StationT['order']))-1
    erosion_out = erosion.copy()
    cc_out = cc.copy()
    name_out = name.copy()
    x_out = x.copy()
    Fertility_out = Fertility.copy()
    erosion_out[order,:] = erosion
    cc_out[order,:] = cc
    name_out[order] = name
    x_out[order,:] = x
    Fertility_out[order] = Fertility
    
    return erosion_out, cc_out, name_out, x_out, Fertility_out
